package loquebot.body;

import java.util.Set;
import java.util.HashMap;
import java.util.ArrayList;
import java.util.logging.Logger;

import cz.cuni.pogamut.Client.AgentBody;

import cz.cuni.pogamut.MessageObjects.MessageObject;
import cz.cuni.pogamut.MessageObjects.MessageType;
import cz.cuni.pogamut.MessageObjects.Triple;

import cz.cuni.pogamut.MessageObjects.NavPoint;
import cz.cuni.pogamut.MessageObjects.Reachable;
import cz.cuni.pogamut.MessageObjects.Path;

import loquebot.Main;
import loquebot.util.LoqueUtils;
import loquebot.util.LoqueListener;
import loquebot.util.LoqueRequestID;
import loquebot.memory.LoqueMemory;

/**
 * Responsible for necessary management around traveling requests.
 *
 * <p>Main purpose of this class is to pick the most suitable destination from
 * the given list of possible destinations (that usually means the closest one)
 * and then keep navigating the agent there. In other words, give me a bunch of
 * destinations and you'll visit one of them or all of them in no time.</p>
 *
 * <h4>Future: do not revisit</h4>
 *
 * When the agent visits a pickup spot that is not the current travel location,
 * ut it is waiting in the list of later items to be visited, such spot should
 * be removed from that list. This might help with occasional <i>pointless</i>
 * returning to places the agent has already visited just a few seconds away.
 * Fortunatelly, this glitch does not occur too often in current logic, but we
 * are supposed to seek the nirvana and strive for perfection.. ;)
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class LoqueTravelManager
{
    /**
     * Current travel destination.
     */
    private Triple travelDestination = null;

    /**
     * Maximum amount of time to travel to each location.
     */
    private int travelTimeout = 0;

    /**
     * Other travel destinations to choose from.
     */
    private ArrayList<Triple> travelMore = null;

    /**
     * Whether to keep travelin around other items.
     */
    private boolean travelLoop = false;

    /*========================================================================*/

    /**
     * Initializes new travel request to the given destinations.
     *
     * @param dest Already chosen, directly reachable destination. Use null,
     * when no destination was chosen directly and the choice is to be made
     * from more destinations through the request to engine's path finding.
     * @param more List of other locations to travel to after this location is
     * reached/failed. This list may be empty to indicate travel request to the
     * chosen directly reachable destination only.
     * @param loop Whether to loop other destinations after the first one.
     * @param timeout Maximum amount of time to travel. Use -1 to for unlimited.
     * @return True, if the request was completed successfully. False otherwise.
     */
    public boolean initRequest (Triple dest, ArrayList<Triple> more, boolean loop, int timeout)
    {
        // setup primary travel destination
        travelDestination = dest;
        // setup further travel destinations
        travelMore = more;
        travelLoop = loop;
        // setup traveling timeout
        travelTimeout = timeout;

        // do we have a directly reachable location?
        if (travelDestination != null)
        {
            // init new navigation
            log.config(
                "TravelManager.initRequest(): setting new direct travel location"
                + ", distance " + (int) memory.self.getSpaceDistance (dest)
                + ", timeout " + travelTimeout
            );
            navigator.initDirectNavigation (travelDestination, travelTimeout);
            // and start traveling
            return keepTraveling ();
        }

        // well, let's request some navigation info otherwise
        requestNavigationInfo ();
        return true;
    }

    /*========================================================================*/

    /**
     * Continues traveling to the chosen destination.
     *
     * @return True, if still traveling. False otherwise.
     */
    public boolean keepTraveling ()
    {
        // do we have a destination already?
        if (travelDestination == null)
        {
            // check about navigation info
            if (!receiveNavigationInfo ())
                // not yet.. keep waiting
                return true;

            // did we really succeeed with the info?
            if (travelDestination == null)
                return false;
        }

        // just navigate until not finished
        if (!navigator.keepNavigating ().terminated)
            return true;

        // so, we've failed to navigate? are we going to loop?
        if (travelLoop && (travelMore.size () > 0))
        {
            log.config(
                "TravelManager.keepTraveling(): switching to the next travel destination"
                +", locations left " + travelMore.size ()
            );

            // FUTURE: did we reach the destination successfully?
            // decide, whether to reinclude/forsake upon no failure

            // kill current destination
            travelDestination = null;

            // now, let's request some new navigation info
            requestNavigationInfo ();
            return true;
        }

        return false;
    }

    /*========================================================================*/

    /**
     * Timeout for navigation info requests. Prevents incidental infinite loop.
     */
    private int pathInfoTimeout = 0;
    /**
     * Starting ID of the last navigation info requests.
     */
    private int pathRequestIDStart = 0;

    /**
     * List of reachability responses.
     */
    private HashMap<Integer, Boolean> pathReachables = new HashMap<Integer, Boolean> ();
    /**
     * List of path responses.
     */
    private HashMap<Integer, ArrayList<NavPoint>> pathPaths = new HashMap<Integer, ArrayList<NavPoint>> ();

    /*========================================================================*/

    /**
     * Initiates new navigation info request.
     */
    private void requestNavigationInfo ()
    {
        log.fine("TravelManager.requestNavigationInfo(): requesting navigation info");

        // reset the request timeout to 2 secs
        pathInfoTimeout = (int) (main.logicFrequency * 2);

        // reset request id and collections
        pathRequestIDStart = LoqueRequestID.nextID (travelMore.size ());
        pathReachables.clear ();
        pathPaths.clear ();

        // for each location from the list..
        int pathRequestIDNext = pathRequestIDStart;
        for (Triple loc : travelMore)
        {
            // let's check about direct navigation possibilities
            body.requestReachcheckLocation (pathRequestIDNext, loc, memory.self.getLocation ());

            // and ask for navigation instructions
            body.getPathToLocation (pathRequestIDNext, loc);

            // and increment request id number
            pathRequestIDNext++;
        }
    }

    /*========================================================================*/

    /**
     * Processes the last navigation info request.
     *
     * <p>The method automatically sets the {@link #travelDestination } to the
     * new chosen navigation destination and initializes the navigation, when
     * the navigation info request is completed.</p>
     *
     * @return True, if the last navigation info request was completed.
     */
    private boolean receiveNavigationInfo ()
    {
        // is it taking too long? fail then..
        if (pathInfoTimeout-- <= 0)
        {
            log.fine("TravelManager.receiveNavigationInfo(): timeout");
            return true;
        }

        int chosenKey;

        // keep us thread safe
        synchronized (pathReachables)
        {
            // check, whether all reachable requests are completed
            if (!receiveCheckResponses (pathReachables.keySet ()))
            {
                log.finest("TravelManager.receiveNavigationInfo(): still waiting for reachability info");
                return false;
            }

            // find the closest reachable one..
            chosenKey = receivePickReachable ();
        }

        // did we find a reachable destination?
        if (chosenKey > 0)
        {
            // remove this location from travel destinations
            log.finest("TravelManager.receiveNavigationInfo(): chosen location " + chosenKey);
            travelDestination = travelMore.remove (chosenKey - pathRequestIDStart);

            // and init new navigation
            log.config(
                "TravelManager.receiveNavigationInfo(): setting new direct travel location"
                + ", disance " + memory.self.getSpaceDistance (travelDestination)
                + ", timeout " + travelTimeout
            );
            navigator.initDirectNavigation (travelDestination, travelTimeout);
            return true;
        }

        // keep us thread safe
        synchronized (pathPaths)
        {
            // check, whether all path requests are completed
            if (!receiveCheckResponses (pathPaths.keySet ()))
            {
                log.finest("TravelManager.receiveNavigationInfo(): still waiting for navigation info");
                return false;
            }

            // find the closest reasonable one..
            chosenKey = receivePickPath ();
        }

        // did we find a reasonable path?
        if (chosenKey > 0)
        {
            // remove this location from travel destinations
            log.finest("TravelManager.receiveNavigationInfo(): chosen path " + chosenKey);
            travelDestination = travelMore.remove (chosenKey - pathRequestIDStart);

            // retreive path info
            ArrayList<NavPoint> chosenPth = pathPaths.get(chosenKey);

            // retreive distance info
            int chosenDst = LoqueUtils.distanceAlongPath (
                memory.self.getLocation (),
                chosenPth, travelDestination
            );

            // retreive appropriate timeout
            int timeout = navigator.getTimeout (travelTimeout, chosenDst);

            // and init new navigation
            log.config(
                "TravelManager.receiveNavigationInfo(): setting new path travel destination"
                + ", nodes " + chosenPth.size()
                + ", distance " + chosenDst
                + ", timeout " + timeout
            );
            navigator.initPathNavigation (travelDestination, chosenPth, timeout);
            return true;
        }

        // no paths left..
        log.fine("TravelManager.receiveNavigationInfo(): fail: no valid paths received");
        travelDestination = null;
        return true;
    }

    /*========================================================================*/

    /**
     * Verifies, whether all awaited responses already arrived.
     * @param responses Set of keys of arrived responses.
     * @return True, if all responses have arrived.
     */
    private boolean receiveCheckResponses (Set<Integer> responses)
    {
        // check for overall count
        if (responses.size () < travelMore.size ())
            return false;

        // check for each response
        // note: this is done in reversed order due to optimizations
        for (
            int i = pathRequestIDStart + travelMore.size () - 1;
            i >= pathRequestIDStart; i--
        )
        {
            // do we have this response already?
            if (!responses.contains(i))
                return false;
        }
        return true;
    }

    /*========================================================================*/

    /**
     * Picks the most suitable one among reachability responses.
     * @return Key of the chosen reachability response.
     */
    private int receivePickReachable ()
    {
        int chosenKey = 0;
        int chosenDst = 0;

        // loop through responses
        for (
            int i = pathRequestIDStart + travelMore.size () - 1;
            i >= pathRequestIDStart; i--
        )
        {
            // is this destination reachable?
            if (!pathReachables.get(i))
                continue;

            // compute it's distance
            int dst = (int) memory.self.getSpaceDistance (
                travelMore.get (i - pathRequestIDStart)
            );

            // do we have at least one reachable yet?
            if (chosenKey > 0)
            {
                // approx-compare the distances, randomize upon equal..
                if (LoqueUtils.approxCompare (dst, chosenDst, 500))
                {
                    // choose this one..
                    chosenKey = i;
                    chosenDst = dst;
                }
            }
            // otherwise, make this one our primary target..
            else
            {
                chosenKey = i;
                chosenDst = dst;
            }

            log.finest("TravelManager.receivePickReachable(): got reachable location " + i + " at " + dst);
        }
        return chosenKey;
    }

    /**
     * Picks the most suitable one among path responses.
     * @return Key of the chosen path response.
     */
    private int receivePickPath ()
    {
        int chosenKey = 0;
        int chosenDst = 0;

        // find the closest one..
        for (
            int i = pathRequestIDStart + travelMore.size () - 1;
            i >= pathRequestIDStart;
            i--
        )
        {
            int dst = LoqueUtils.distanceAlongPath (
                memory.self.getLocation (), pathPaths.get(i),
                travelMore.get (i - pathRequestIDStart)
            );

            ArrayList<NavPoint> pth = pathPaths.get(i);

            // does this path make sence?
            // there might come stupid path responses with no nodes
            // but with long distance.. most of them are not valid
            if ( (pth.size () <= 0) && (dst > 1000) )
            {
                log.finest("TravelManager.receivePickPath(): got invalid path " + i + " with " + pth.size() + " nodes " + " and length " + dst);
                continue;
            }

            // do we have at least one reachable yet?
            if (chosenKey > 0)
            {
                // approx-compare the distances, randomize upon equal..
                if (LoqueUtils.approxCompare (dst, chosenDst, 500))
                {
                    // choose this one..
                    chosenKey = i;
                    chosenDst = dst;
                }
            }
            // otherwise, make this one our primary target..
            else
            {
                chosenKey = i;
                chosenDst = dst;
            }

            log.finest("TravelManager.receivePickPath(): got path " + i + " with " + pth.size() + " nodes " + " and length " + dst);
        }
        return chosenKey;
    }

    /*========================================================================*/

    /**
     * Listening class for messages from engine.
     */
    private class Listener extends LoqueListener
    {
        /**
         * Agent received reachability response.
         * @param msg Message to handle.
         */
        private void msgReachable (Reachable msg)
        {
            // parse the response id
            int ID = Integer.parseInt(msg.pongID);
            synchronized (pathReachables)
            {
                // is it an obsolete request?
                if (ID < pathRequestIDStart) return;
                // process this request
                pathReachables.put(ID, msg.reachable);
            }
            //log.config("TravelManager.Listener: reachability response recevived: " + ID);
        }

        /**
         * Agent received path info response.
         * @param msg Message to handle.
         */
        private void msgPath (Path msg)
        {
            // parse the response id
            int ID = Integer.parseInt(msg.pongID);
            synchronized (pathPaths)
            {
                // is it an obsolete request?
                if (ID < pathRequestIDStart) return;
                // process this request
                pathPaths.put(ID, msg.nodes);
            }
            //log.config("TravelManager.Listener: path response recevived: " + ID);
        }

        /**
         * Message switch.
         * @param msg Message to handle.
         */
        protected void processMessage (MessageObject msg)
        {
            switch (msg.type)
            {
                case REACHABLE:
                    msgReachable ((Reachable) msg);
                    return;
                case PATH:
                    msgPath ((Path) msg);
                    return;
            }
        }

        /**
         * Constructor: Signs up for listening.
         */
        private Listener ()
        {
            body.addTypedRcvMsgListener(this, MessageType.PATH);
            body.addTypedRcvMsgListener(this, MessageType.REACHABLE);
        }
    }

    /** Listener. */
    private LoqueListener listener;

    /*========================================================================*/

    /**
     * Loque Navigator.
     */
    private LoqueNavigator navigator;

    /*========================================================================*/

    /** Agent's main. */
    protected Main main;
    /** Loque memory. */
    protected LoqueMemory memory;
    /** Agent's body. */
    protected AgentBody body;
    /** Agent's log. */
    protected Logger log;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     */
    public LoqueTravelManager (Main main, LoqueMemory memory)
    {
        // setup reference to agent
        this.main = main;
        this.memory = memory;
        this.body = main.getBody ();
        this.log = main.getLogger ();

        // create runner object
        this.navigator = new LoqueNavigator (main, memory);

        // create listener
        this.listener = new Listener ();
    }
}